/* searcher.c - <comment goes here> */
/* Created: Wed 21 Jun 2023 09:54:25 PM EDT malakai */
/* $Id: $ */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <glib.h>

#include "searcher.h"

/* local #defines */
#define POS_NONE 0
#define POS_X 1
#define POS_Y 2
#define POS_Z 4
#define POS_ALL (POS_X|POS_Y|POS_Z)

#define TESTFILE "test.nec"
#define CSVFILE "test.csv"
#define S2PFILE "test-maxgain.s2p"
#define XNECPROCESSES 8

/* structs and typedefs */

struct endpoint {
	double x;
	double y;
	double z;
	int fix;
	int inherit;
	int tag;
};

struct design {
	GList *uhf_section;
	double bracket_half_width;
	double wire_radius;
	double narod_length;
	double narod_coupling_distance;
	double xmin,ymin,zmin; /* tweak constraint */
	double xmax,ymax,zmax; /* tweak constraint */
	double score;
};

/* local variable declarations */
//double bracket_half_width = 2.2e-02;
//double wire_radius = 1.65e-03;
int feed_tag = 20;
int narod_tag_start = 10;
int support_tag = 30;
/* fewer segments runs faster, with less precision.  mult of 50 seems like reasonable tradeoff. */
//double autosegment_mult = 100.0;
double autosegment_mult = 50.0;

/* local function declarations */
int snprintf_inches(char *buf, size_t size,double inches, int base);
int snprintf_measure(char *buf, size_t size,double inches, int base);
double meters_to_inches(double m);


/* code starts here. */

double segment_length(
	double x1, double y1, double z1,
	double x2, double y2, double z2
) {

	double x,y,z;

	x = x2-x1;
	x = x*x;
	y = y2-y1;
	y = y*y;
	z = z2-z1;
	z = z*z;
	return(sqrt(x+y+z));
}

int autosegment(double length) {
	int segs = length*autosegment_mult;
	return((segs>0)?segs:1);
}

void nec_wire(FILE *out, int tag, int segments, 
	double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	double wire_radius
) {

	if(segments <=0 ) {
		segments = autosegment(segment_length(x1,y1,z1,x2,y2,z2));
	}
	fprintf(out,"GW %d %d %e %e %e %e %e %e %e\n",
		tag,segments,
		x1,y1,z1,
		x2,y2,z2,
		wire_radius
	);
}

void nec_arc(FILE *out, int tag, int segments,
	double yrad, double e1angle, double e2angle,
	double wire_radius
) {
	if(segments <=0 ) {
		segments = autosegment(yrad*10.0);
	}
	fprintf(out,"GA %d %d %e %e %e %e 0 0 0\n",
		tag,segments,
		yrad,e1angle,e2angle,
		wire_radius
	);
}

void nec_section_cm(FILE* out,struct design *d) {
	
	fprintf(out,"' Comments\n");
	fprintf(out,"CM Score=%f\n",d->score);
	fprintf(out,"CE\n");

}

void nec_section_uhf(FILE* out,struct design *d) {

	struct endpoint *from;
	struct endpoint *to;
	GList *bend;
	double x0,y0,z0,x1,y1,z1;
	char buf[8192];
	double halflength = 0.0;

	fprintf(out,"' UHF\n");

	from = d->uhf_section->data;
	x0 = from->x;
	y0 = from->y;
	z0 = from->z;

	for(bend = d->uhf_section; bend && bend->next; bend = bend->next) {
		from = bend->data;
		to = bend->next->data;

		x1 = to->x;
		y1 = to->y;
		z1 = to->z;

		if(to->inherit & POS_X) x1 = x0;
		if(to->inherit & POS_Y) y1 = y0;
		if(to->inherit & POS_Z) z1 = z0;
	
		double seglen = segment_length(x0,y0,z0,x1,y1,z1);
		halflength += seglen;
		int segs = autosegment(seglen);

		snprintf_measure(buf,sizeof(buf),seglen,16);
		fprintf(out,"' uhf segment %d length = %s\n",to->tag,buf);
		snprintf_measure(buf,sizeof(buf),z1,16);
		fprintf(out,"' uhf segment %d height = %s\n",to->tag,buf);
		fprintf(out,"GW %d %d %e %e %e %e %e %e %e\n",
			to->tag, segs,
			x0,y0,z0,
			x1,y1,z1,
			(d->wire_radius)
		);
		x0=x1;
		y0=y1;
		z0=z1;
	}
	snprintf_measure(buf,sizeof(buf),2.0*halflength,16);
	fprintf(out,"' UHF side LOA %s\n",buf);

}

void nec_section_vhf(FILE* out, struct design *d) {

	GList *l;
	struct endpoint *e;
	double uhfzmax = 0.0;
	double uhfxmax = 0.0;
	int tag = narod_tag_start;
	char buf[8192];

	/* find the highest z point in the uhf design. */

	for(l=d->uhf_section;l;l=l->next) {
		e=l->data;
		if( (e->z > uhfzmax) &&
			(!(e->inherit & POS_Z))
		) {
			uhfzmax = e->z;
		}
		if( (e->x > uhfxmax) &&
			(!(e->inherit & POS_X))
		) {
			uhfxmax = e->x;
		}
	}
	snprintf_measure(buf,sizeof(buf),uhfzmax+d->narod_coupling_distance,16);
	fprintf(out,"' VHF narod top height = %s\n",buf);

	snprintf_measure(buf,sizeof(buf),d->narod_length,16);
	if(d->narod_length < uhfxmax) {
		fprintf(out,"' VHF narod length %s no wrapping\n",buf);
		nec_wire(out,
			tag++,-1,
			-d->bracket_half_width,0.0,uhfzmax+d->narod_coupling_distance,
			d->narod_length,0.0,uhfzmax+d->narod_coupling_distance,
			(d->wire_radius)
		);
	} else {
		double topseglen = uhfxmax+d->bracket_half_width;
		snprintf_measure(buf,sizeof(buf),topseglen,16);

		fprintf(out,"' Top Segment %d length to wrap %s\n", tag, buf);
		nec_wire(out,
			tag++,-1,
			-d->bracket_half_width,0.0,uhfzmax+d->narod_coupling_distance,
			uhfxmax,0.0,uhfzmax+d->narod_coupling_distance,
			(d->wire_radius)
		);
		nec_arc(out,
			tag++,15,
			d->narod_coupling_distance,90,-90,
			(d->wire_radius)
		);
		fprintf(out,"GM 0 0 0 0 0 %e 0 %e %d\n",
			uhfxmax,uhfzmax,tag-1
		);

		double tail_length = d->narod_length - (-d->bracket_half_width + uhfxmax);
		double ncl = 0.0;
		e = g_list_nth_data(d->uhf_section,
				g_list_length(d->uhf_section)-1
			);
		ncl = e->x;
		e = g_list_nth_data(d->uhf_section,
				g_list_length(d->uhf_section)-2
			);
		ncl = ncl - e->x;
		ncl -= d->narod_coupling_distance;
		if(tail_length > ncl) {
			tail_length = ncl;
		}

		snprintf_measure(buf,sizeof(buf),tail_length,16);
		fprintf(out,"' Tail Segment after wrap %d %s\n", tag,buf);
		nec_wire(out,
			tag++,-1,
			uhfxmax,0.0,uhfzmax-d->narod_coupling_distance,
			(uhfxmax-tail_length),0.0,uhfzmax-d->narod_coupling_distance,
			(d->wire_radius)
		);

		/* force the narod length in the design to conform to reality. */
		d->narod_length = 
			topseglen +
			(M_PI * d->narod_coupling_distance) +
			tail_length;
		snprintf_measure(buf,sizeof(buf),d->narod_length,16);
		fprintf(out,"' VHF narod length %s with wraparound\n",buf);
	}
	snprintf_measure(buf,sizeof(buf),2.0*d->narod_length,16);
	fprintf(out,"' VHF narod LOA %s\n",buf);

	/* the outbound chunk */

}

void nec_section_symmetry(FILE* out, struct design *d) {

	int start_from_tag = 1;
	
	fprintf(out,"' Symmetry\n");
	/* move out to bracket halfwidth. */
	fprintf(out,"GM 0 0 0 0 0 %e 0 0 %d\n",
		d->bracket_half_width,
		start_from_tag
	);
	/* xy and yz reflection. */
	fprintf(out,"GX 0 101 0 0 0 0 0 0 0\n");
}

void nec_section_feed(FILE* out, struct design *d) {

	char buf[8192];

	snprintf_measure(buf,sizeof(buf),2.0*d->bracket_half_width,16);
	fprintf(out,"' Feed width %s\n",buf);
	nec_wire(out,
		feed_tag,1,
		d->bracket_half_width,0.0,0.0,
		-d->bracket_half_width,0.0,0.0,
		(d->wire_radius)
	);
	fprintf(out,"GE 0 0 0 0 0 0 0 0 0\n");
	fprintf(out,"EX 0 %d 1 0 0 0 0 0 0\n",feed_tag);
	fprintf(out,"ZO 300 0 0 0 0 0 0 0 0 0\n");
}

void nec_section_scans(FILE* out) {
	fprintf(out,"' VHF Scan\n");
	fprintf(out,"FR 0 7 0 0 %e %e %e 0 0 0\n",
		177.0,6.0,213.0
	);
	fprintf(out,"' UHF Scan\n");
	fprintf(out,"FR 0 23 0 0 %e %e %e 0 0 0\n",
		473.0,6.0,605.0
	);
	fprintf(out,"RP 0 38 73 0 0 0 5 5 0 0\n");

}
void nec_section_end(FILE* out) {
	fprintf(out,"' End\n");
	fprintf(out,"EN 0 0 0 0 0 0 0 0 0\n");
}

void nec_section_boundarypin(FILE* out, struct design *d) {

	nec_wire(out,
		100,1,
		d->xmax,d->ymax,d->zmax,
		d->xmax+0.001,d->ymax+0.001,d->zmax+0.001,
		0.001
	);

}

void nec_section_bolts(FILE *out,struct design *d, int *tag, double height) {

	double longbolt = 3.175e-02;
	double inchbolt = 2.54e-02;
	double shortbolt = 1.27e-02;
	double nobolt = 0.0;

	double bolt_A_len = inchbolt;
	double bolt_B_len = inchbolt;
	double bolt_C_len = nobolt;

	double bolt_A_rad = 2.413e-03;
	double bolt_B_rad = 2.413e-03;
	double bolt_C_rad = 2.413e-03 * 2.0;
	int t= *tag;
	char buf[8192];

	if(bolt_A_len > 0.0) {
		snprintf_measure(buf,sizeof(buf),bolt_A_len,16);
		fprintf(out,"'   bolt A %s\n",buf);
		nec_wire(out,t++,-1,
			-d->bracket_half_width,0.0,height,
			-d->bracket_half_width,-bolt_A_len,height,
			bolt_A_rad
		);
	}
	if(bolt_B_len > 0.0) {
		snprintf_measure(buf,sizeof(buf),bolt_B_len,16);
		fprintf(out,"'   bolt B %s\n",buf);
		nec_wire(out,t++,-1,
			d->bracket_half_width,0.0,height,
			d->bracket_half_width,-bolt_B_len,height,
			bolt_B_rad
		);
	}
	if(bolt_C_len > 0.0) {
		snprintf_measure(buf,sizeof(buf),bolt_C_len,16);
		fprintf(out,"'   bolt C %s\n",buf);
		nec_wire(out,t++,-1,
			0.0,0.0,height,
			0.0,-bolt_C_len,height,
			bolt_C_rad
		);
	}

	*tag = t;
}

/* add support bolt information to the design. */
void nec_section_supports(FILE *out,struct design *d) {

	struct endpoint *e;
	double narod_height;
	double support_height;
	int tag = support_tag;

	fprintf(out,"' Support bolts\n");
	fprintf(out,"'   Narod Support\n");

	e = g_list_nth_data(d->uhf_section,
			g_list_length(d->uhf_section)-2
	);
	narod_height = e->z + d->narod_coupling_distance;
	nec_section_bolts(out,d,&tag,narod_height);
	nec_section_bolts(out,d,&tag,-narod_height);

	/* 2? it'd be better to have the support points marked in the uhf design, but
	 * oh well. */
	fprintf(out,"'   Middle Support\n");
	e = g_list_nth_data(d->uhf_section,
			2 
	);
	support_height = e->z;
	nec_section_bolts(out,d,&tag,support_height);
	nec_section_bolts(out,d,&tag,-support_height);

	fprintf(out,"'   Feed Support\n");
	nec_section_bolts(out,d,&tag,0.0);

}


void nec_print(char *filename,struct design *d) {

	FILE *out;

	if(!(out = fopen(filename,"w"))) {
		perror("Ooops.");
		exit(0);
	}

	nec_section_cm(out,d);   
	nec_section_uhf(out,d);  /*will need pointers to structure. */
	nec_section_vhf(out,d);
	//nec_section_boundarypin(out,d);
	nec_section_symmetry(out,d);
	nec_section_supports(out,d);
	nec_section_feed(out,d);
	nec_section_scans(out);
	nec_section_end(out);

	fclose(out);
}

struct endpoint *new_endpoint(void) {
	struct endpoint *new;

	new = (struct endpoint *)g_malloc(sizeof(struct endpoint));
	new->x = new->y = new->z = 0.0;
	new->fix = POS_NONE;
	new->inherit = POS_NONE;
	new->tag = 0;
	return(new);

}

struct endpoint *dup_endpoint(struct endpoint *a) {
	struct endpoint *new;

	new = (struct endpoint *)g_malloc(sizeof(struct endpoint));
	new->x = a->x;
	new->y = a->y;
	new->z = a->z;
	new->fix = a->fix;
	new->inherit = a->inherit;
	new->tag = a->tag;
	return(new);

}

int run_xnec(char *input_filename, char *output_filename) {

	char buf[8192];

	sprintf(buf,"/usr/local/bin/xnec2c -b -j %d --write-csv %s -i %s > /dev/null 2>&1", 
		XNECPROCESSES,
		output_filename,
		input_filename
	);

	return(system(buf));
}

void free_endpoint(struct endpoint *f) {
	if(f) g_free(f);
}

/* to count, mhz must be >= mhzmin, <= mhzmax */
double avg_gain(char *csv_filename,double mhzmin, double mhzmax) {

	FILE *in;
	char line[8192];
	double gainmin = 1e8;
	double gainmax = 0.0;
	double mhz,gain_net;
	int fields;

	if(!(in = fopen(csv_filename,"r"))) {
		perror("reading csv");
		exit(0);
	}

	/* discard the first line, it is headers. */
	fgets(line,sizeof(line),in);

	/* assumes mhz is first number, gain_net is 12th. */
	while(fgets(line,sizeof(line),in)) {
		fields = sscanf(line,"%lf,%*lf,%*lf,%*lf,%*lf,%*lf,%*lf,%*lf,%*lf,%*lf,%*lf,%lf",
			&mhz,&gain_net
		);
		if((mhz >= mhzmin) && (mhz <= mhzmax)) {
			if(gain_net < gainmin) {
				gainmin = gain_net;
			}
			if(gain_net > gainmax) {
				gainmax = gain_net;
			}
		}
	};
	fclose(in);
	return( (gainmin + gainmax) / 2.0 );


}

/* weighted average of vhf avg net gain and uhf net gain. */
double score_design_v1(char *csv_filename) {

	double vhfavggain = avg_gain(csv_filename,174,216);
	double uhfavggain = avg_gain(csv_filename,470,608);
	double score = ( ((7.0/30.0)* vhfavggain) + ((23.0/30.0) * uhfavggain));

	/*
	fprintf(stdout,"design score= %lf (VHF avg net gain= %lf, UHF avg net gain = %lf\n",
		score,
		vhfavggain,
		uhfavggain
	);
	*/
	
	return(score);

}

/* weighted average of vhf avg net gain and uhf net gain. */
double score_design_v2(char *csv_filename) {

	double vhfavggain = avg_gain(csv_filename,174,216);
	if((vhfavggain <= 2.0) || (vhfavggain >= 40.0)) return(0.0);

	double uhfavggain = avg_gain(csv_filename,470,608);
	if((uhfavggain <= 2.0) || (uhfavggain >= 40.0)) return(0.0);

	double score = ( ((7.0/30.0)* vhfavggain) + ((23.0/30.0) * uhfavggain));

	return(score);

}

double score_design_uhfnetgain(char *csv_filename) {

	double uhfavggain = avg_gain(csv_filename,470,608);
	
	return(uhfavggain);

}

double score_design_vhfnetgain(char *csv_filename) {

	double uhfavggain = avg_gain(csv_filename,174,216);
	
	return(uhfavggain);

}

struct design *new_design(void) {
	struct design *new;

	new = (struct design *)g_malloc(sizeof(struct design));

	new->narod_length = 0.0;
	new->narod_coupling_distance = 0.0;
	new->bracket_half_width = 2.5e-02;
	new->wire_radius = 1.65e-03;
	new->score = 0.0;
	new->xmin = new->ymin = new->zmin = 0.0;
	new->xmax = 0.5; 	/* half meter. */
	new->ymax = 0.5; 	/* half meter. */
	new->zmax = 0.5; 	/* half meter. */

	new->uhf_section = NULL;

	return(new);
}

struct design *dup_design(struct design *a) {
	struct design *new;
	struct enpoint *ep;

	new = (struct design *)g_malloc(sizeof(struct design));

	/* 
	new->narod_length = a->narod_length;
	new->narod_coupling_distance = a->narod_coupling_distance;
	new->score = a->score;
	new->xmin = a->xmin;
	new->ymin = a->ymin;
	new->zmin = a->zmin;
	new->xmax = a->xmax;
	new->ymax = a->ymax;
	new->zmax = a->zmax;
	*/
	*new = *a;
	new->uhf_section = NULL;

	for(GList *i = a->uhf_section;i;i=i->next) {
		new->uhf_section = g_list_append(new->uhf_section,dup_endpoint(i->data));
	}

	return(new);
}

void free_design(struct design *f) {
	GList *i,*n;

	if(f->uhf_section) {
		g_list_free_full (g_steal_pointer (&(f->uhf_section)), (GDestroyNotify) free_endpoint);
	}
	g_free(f);

}


double score_design(char *csv_filename) {
	//return(score_design_uhfnetgain(csv_filename));
	//return(score_design_vhfnetgain(csv_filename));
	return(score_design_v2(csv_filename));
}

int tweak_design_vhf(struct design *d,double temp) {

	//int magnitude = ((rand()%16)*temp)+1;
	int magnitude = ((rand()%16)+1)*temp;
	int direction = (rand()%2)?1:-1;
	double minstep = 0.003175;  /* 1/8in */
	double modification = (minstep*direction*magnitude);

	if((d->narod_length + modification) <= d->xmin) return(0);
	if((d->narod_length + modification) >= d->xmax) return(0);

	//fprintf(stdout,"tweaking narod length %f by %f\n",d->narod_length,modification);

	d->narod_length += modification;

	return(1);
}

int tweak_design_uhf(struct design *d,double temp) {
	
	int uhf_sections;
	int modify;
	int addsub;
	struct endpoint *e;
	
	uhf_sections = g_list_length(d->uhf_section);
	modify = (rand()%(uhf_sections-1))+1;
	e = g_list_nth_data(d->uhf_section,modify);

	if(e->fix == POS_ALL) return(0);

	int position = rand()%3;
	int ng = 0;
	
	switch(position) {
		default:
		case 0:
			if(e->fix & POS_X) ng++;
			if(e->inherit & POS_X) ng++;
			break;
		case 1:
			if(e->fix & POS_Y) ng++;
			if(e->inherit & POS_Y) ng++;
			break;
		case 2:
			if(e->fix & POS_Z) ng++;
			if(e->inherit & POS_Z) ng++;
			break;
	}
	if(ng) {
		return(0);
	}

	//int magnitude = ((rand()%16)*temp)+1;
	int magnitude = ((rand()%16)+1)*temp;
	int direction = (rand()%2)?1:-1;
	double minstep = 0.003175;  /* 1/8in */
	double modification = (minstep*direction*magnitude);
	int mod = 0;

	switch(position) {
		default:
		case 0:
			if((e->x + modification) >= d->xmax) break;
			if((e->x + modification) <= d->xmin) break;
			e->x = e->x + modification;
			mod++;
			break;
		case 1:
			if((e->y + modification) >= d->ymax) break;
			if((e->y + modification) <= d->ymin) break;
			e->y = e->y + modification;
			mod++;
			break;
		case 2:
			if((e->z + modification) >= d->zmax) break;
			if((e->z + modification) <= d->zmin) break;
			e->z = e->z + modification;
			mod++;
			break;
	}

	if(mod) {
		return(1);
	} else {
		return(0); 
	}

}

/* apply some small variations  onto the design. */
int tweak_design(struct design *d,double temp) {
	if(rand()%5 == 0) {
		for(int i=0;i<1000;i++) {
			if(tweak_design_vhf(d,temp)) return(1);
		}
	} else {
		for(int i=0;i<1000;i++) {
			if(tweak_design_uhf(d,temp)) return(1);
		}
	}
	return(0);
}

/* Jeffrika#6 is the hoverman variation that I came up with 'by hand'. It was
 * actually several iterations of spreadsheet to manual xnec2c run, with tweaks
 * to the UHF and VHF wire lengths. Its a pretty good antenna design IMHO.  :)
 * its a great starting design, but I think we can do better!*/
struct design *jeffrika_6(void) {

	struct design *id; /* initial design */
	struct endpoint *ep;
	int tag = 0;

	id = new_design();

	id->narod_length = 0.390;  /* from jeffrika6 */
	id->narod_coupling_distance = 0.0127;  /* half inch, from jeffrika6 */

	/* fill out the uhf control points from jeffrika6 */
	/* the feed point. */
	ep = new_endpoint();
	ep->fix = POS_ALL;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* first bend */
	ep = new_endpoint();
	ep->x = 1.34602e-01;
	ep->z = 1.34602e-01;
	ep->fix = POS_Y;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* back to support */
	ep = new_endpoint();
	ep->x = 0.0;
	ep->z = 2.69203e-01;
	ep->fix = POS_X|POS_Y;  /* distance to center is also fixed! */
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* out again */
	ep = new_endpoint();
	ep->x = 1.34602e-01;
	ep->z = 4.03805E-01;
	ep->fix = POS_Y;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* flat section for narod coupling */
	ep = new_endpoint();
	ep->x = 3.24957E-01;
	ep->fix = POS_Y;
	ep->inherit = POS_Z; /* endpoint cant move up or down. */
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	return(id);

}

/* This is an optimized design against score_design_v2 that used jeffrika_6 as
 * initial input, and was found by this program.  I did a few different
 * variations on iterations, bad-run lengths, and slightly different search
 * algorithms.  This design (or near neighbors to it) kept coming up, despite
 * the algorithm changes, so I think its probably pretty good.  I'm codifiying
 * it as an additional starting design for future algorithmic search
 * variations, and to print out the design parameters in inches, cause I don't
 * own a metric tape measure.  That's ona counna 'murica. */
struct design *jeffrika_7(void) {

	struct design *id; /* initial design */
	struct endpoint *ep;
	int tag = 0;

	id = new_design();

	id->narod_length = 3.199250e-01;  
	id->narod_coupling_distance = 0.0127;  /* half inch, from jeffrika6 */

	/* fill out the uhf control points from jeffrika6 */
	/* the feed point. */
	ep = new_endpoint();
	ep->fix = POS_ALL;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* first bend */
	ep = new_endpoint();
	ep->x = 1.949270e-01;
	ep->z = 3.617700e-02;
	ep->fix = POS_Y;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* back to support */
	ep = new_endpoint();
	ep->x = 0.0;
	ep->z = 3.200030e-01;
	ep->fix = POS_X|POS_Y;  /* distance to center is also fixed! */
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* out again */
	ep = new_endpoint();
	ep->x = 1.854020e-01;
	ep->z = 4.165050e-01;
	ep->fix = POS_Y;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* flat section for narod coupling */
	ep = new_endpoint();
	ep->x = 2.773320e-01;
	ep->fix = POS_Y;
	ep->inherit = POS_Z; /* endpoint cant move up or down. */
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	return(id);

}

struct design *jeffrika_7b(void) {

	struct design *id; /* initial design */
	struct endpoint *ep;
	int tag = 0;

	id = new_design();

	id->narod_length = 4.471500e-01;
	id->narod_coupling_distance = 0.0127;  /* half inch, from jeffrika6 */

	/* fill out the uhf control points from jeffrika6 */
	/* the feed point. */
	ep = new_endpoint();
	ep->fix = POS_ALL;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* first bend */
	ep = new_endpoint();
	ep->x = 1.917520e-01;
	ep->z = 5.840200e-02;
	ep->fix = POS_Y;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* back to support */
	ep = new_endpoint();
	ep->x = 0.0;
	ep->z = 3.327030e-01;
	ep->fix = POS_X|POS_Y;  /* distance to center is also fixed! */
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* out again */
	ep = new_endpoint();
	ep->x = 1.822270e-01;
	ep->z = 4.196800e-01;
	ep->fix = POS_Y;
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	/* flat section for narod coupling */
	ep = new_endpoint();
	ep->x = 2.678070e-01;
	ep->fix = POS_Y;
	ep->inherit = POS_Z; /* endpoint cant move up or down. */
	ep->tag = tag++;
	id->uhf_section = g_list_append(id->uhf_section,ep);

	return(id);

}


double meters_to_inches(double m) {
	return(m / 0.0254);
}

int snprintf_inches(char *buf, size_t size,double inches, int base) {

	int in = inches;
	int ths = round((inches - in) * base);
	int ret;

	while( (base > 0) && ((ths & 1) == 0) ) {
		ths= ths >>1;
		base = base>>1;
	}
	if((ths == 1) && (base == 1)) {
		in++;
		base =0;
	}
	if(base) {
		ret = g_snprintf(buf,size,"%d+%d/%d\"",in,ths,base);
	} else {
		ret = g_snprintf(buf,size,"%d\"",in);
	}
	return(ret);

}

int snprintf_measure(char *buf, size_t size,double meters, int base) {

	char *s = buf;
	char *eos = buf+size;

	s += g_snprintf(s,eos-s,"%dmm ",(int)round(meters*1000));
	s += snprintf_inches(s,eos-s,meters_to_inches(meters),16);
	return(s-buf);

}


int main(int argc, char** argv) {
	
	struct design *id; /* initial design */
	struct design *td; /* test design */
	struct design *nd; /* neighboring design */
	struct design *bd; /* best design */
	double score;
	double K;

	unsigned int seed = time(0);

	GList *uhf = NULL;


	srand(seed);
	srand48(seed);

	//id = jeffrika_6();
	//id = jeffrika_7();
	id = jeffrika_7b();

	nec_print(TESTFILE,id);
	run_xnec(TESTFILE,CSVFILE);
	id->score = score_design(CSVFILE);
	fprintf(stdout,"Initial Design Score: %lf\n",id->score);
	fprintf(stdout,"Saving in initial.nec\n");
	nec_print("initial.nec",id);

	bd = dup_design(id);
	nec_print("best.nec",bd);
	td = dup_design(id);

	int iterations = 150;
	int maxprobe = 30;
	/* found jeffrika_7 from jeffrika_6 */
	/*int iterations = 3600; */
	/*int maxprobe = 60;	 */ 
	int sincewinner =0;

	for(int i=iterations;i>=0;i--) {
		fprintf(stdout,"%d: best score %f, test score %f\n",i,bd->score,td->score);
		nd = dup_design(td);
		K = ((double)i/iterations);
		tweak_design(nd,K);
		nec_print(TESTFILE,nd);
		run_xnec(TESTFILE,CSVFILE);
		nd->score = score_design(CSVFILE);
		//fprintf(stdout,"\tNeighboring Design Score: %lf\n",nd->score);
		if(nd->score > bd->score) {
			free_design(bd);
			bd = dup_design(nd);
			fprintf(stdout,"\t>>> Thats a winner! Saving in best.nec ***\n");
			nec_print("best.nec",bd);
			sincewinner = 0;
		} else {
			sincewinner++;
		}
		int accept = 0;
		double cutoff = (2.0 * K)*drand48();
		if( (nd->score >= td->score)) { 
			fprintf(stdout,"\t+++ improved score, delta %f\n",(nd->score - td->score));
			accept++;
		} else if ((nd->score > 0.0) && (fabs(nd->score - td->score) < (cutoff))) {
			fprintf(stdout,"\t--- allowing worse score, delta %f beat cutoff %f\n",
				(nd->score - td->score),
				cutoff
			);
			accept++;
		} else {
			fprintf(stdout,"\t... delta %f didn't beat cutoff %f\n",
				(nd->score - td->score),
				cutoff
			);
		}
		if (sincewinner >= maxprobe) {
			free_design(td);
			td = dup_design(bd);
			fprintf(stdout,"\t<<< going back to best design.\n");
			sincewinner = 0;
		} else if(accept) {
			free_design(td);
			td = dup_design(nd);
		}
			
		free_design(nd);
	}
	fprintf(stdout,"Best Score: %lf  see best.nec.\n",bd->score);
}
